﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codeconnections/CodeConnectionsRequest.h>
#include <aws/codeconnections/CodeConnections_EXPORTS.h>
#include <aws/codeconnections/model/VpcConfiguration.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CodeConnections {
namespace Model {

/**
 */
class UpdateHostRequest : public CodeConnectionsRequest {
 public:
  AWS_CODECONNECTIONS_API UpdateHostRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateHost"; }

  AWS_CODECONNECTIONS_API Aws::String SerializePayload() const override;

  AWS_CODECONNECTIONS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the host to be updated.</p>
   */
  inline const Aws::String& GetHostArn() const { return m_hostArn; }
  inline bool HostArnHasBeenSet() const { return m_hostArnHasBeenSet; }
  template <typename HostArnT = Aws::String>
  void SetHostArn(HostArnT&& value) {
    m_hostArnHasBeenSet = true;
    m_hostArn = std::forward<HostArnT>(value);
  }
  template <typename HostArnT = Aws::String>
  UpdateHostRequest& WithHostArn(HostArnT&& value) {
    SetHostArn(std::forward<HostArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The URL or endpoint of the host to be updated.</p>
   */
  inline const Aws::String& GetProviderEndpoint() const { return m_providerEndpoint; }
  inline bool ProviderEndpointHasBeenSet() const { return m_providerEndpointHasBeenSet; }
  template <typename ProviderEndpointT = Aws::String>
  void SetProviderEndpoint(ProviderEndpointT&& value) {
    m_providerEndpointHasBeenSet = true;
    m_providerEndpoint = std::forward<ProviderEndpointT>(value);
  }
  template <typename ProviderEndpointT = Aws::String>
  UpdateHostRequest& WithProviderEndpoint(ProviderEndpointT&& value) {
    SetProviderEndpoint(std::forward<ProviderEndpointT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The VPC configuration of the host to be updated. A VPC must be configured and
   * the infrastructure to be represented by the host must already be connected to
   * the VPC.</p>
   */
  inline const VpcConfiguration& GetVpcConfiguration() const { return m_vpcConfiguration; }
  inline bool VpcConfigurationHasBeenSet() const { return m_vpcConfigurationHasBeenSet; }
  template <typename VpcConfigurationT = VpcConfiguration>
  void SetVpcConfiguration(VpcConfigurationT&& value) {
    m_vpcConfigurationHasBeenSet = true;
    m_vpcConfiguration = std::forward<VpcConfigurationT>(value);
  }
  template <typename VpcConfigurationT = VpcConfiguration>
  UpdateHostRequest& WithVpcConfiguration(VpcConfigurationT&& value) {
    SetVpcConfiguration(std::forward<VpcConfigurationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_hostArn;
  bool m_hostArnHasBeenSet = false;

  Aws::String m_providerEndpoint;
  bool m_providerEndpointHasBeenSet = false;

  VpcConfiguration m_vpcConfiguration;
  bool m_vpcConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeConnections
}  // namespace Aws
