﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/awstransfer/TransferRequest.h>
#include <aws/awstransfer/Transfer_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Transfer {
namespace Model {

/**
 */
class ListUsersRequest : public TransferRequest {
 public:
  AWS_TRANSFER_API ListUsersRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListUsers"; }

  AWS_TRANSFER_API Aws::String SerializePayload() const override;

  AWS_TRANSFER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Specifies the number of users to return as a response to the
   * <code>ListUsers</code> request.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListUsersRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If there are additional results from the <code>ListUsers</code> call, a
   * <code>NextToken</code> parameter is returned in the output. You can then pass
   * the <code>NextToken</code> to a subsequent <code>ListUsers</code> command, to
   * continue listing additional users.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListUsersRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A system-assigned unique identifier for a server that has users assigned to
   * it.</p>
   */
  inline const Aws::String& GetServerId() const { return m_serverId; }
  inline bool ServerIdHasBeenSet() const { return m_serverIdHasBeenSet; }
  template <typename ServerIdT = Aws::String>
  void SetServerId(ServerIdT&& value) {
    m_serverIdHasBeenSet = true;
    m_serverId = std::forward<ServerIdT>(value);
  }
  template <typename ServerIdT = Aws::String>
  ListUsersRequest& WithServerId(ServerIdT&& value) {
    SetServerId(std::forward<ServerIdT>(value));
    return *this;
  }
  ///@}
 private:
  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  Aws::String m_serverId;
  bool m_serverIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace Transfer
}  // namespace Aws
