﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iam/IAMRequest.h>
#include <aws/iam/IAM_EXPORTS.h>

#include <utility>

namespace Aws {
namespace IAM {
namespace Model {

/**
 */
class PutUserPolicyRequest : public IAMRequest {
 public:
  AWS_IAM_API PutUserPolicyRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PutUserPolicy"; }

  AWS_IAM_API Aws::String SerializePayload() const override;

 protected:
  AWS_IAM_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the user to associate the policy with.</p> <p>This parameter
   * allows (through its <a href="http://wikipedia.org/wiki/regex">regex pattern</a>)
   * a string of characters consisting of upper and lowercase alphanumeric characters
   * with no spaces. You can also include any of the following characters:
   * _+=,.@-</p>
   */
  inline const Aws::String& GetUserName() const { return m_userName; }
  inline bool UserNameHasBeenSet() const { return m_userNameHasBeenSet; }
  template <typename UserNameT = Aws::String>
  void SetUserName(UserNameT&& value) {
    m_userNameHasBeenSet = true;
    m_userName = std::forward<UserNameT>(value);
  }
  template <typename UserNameT = Aws::String>
  PutUserPolicyRequest& WithUserName(UserNameT&& value) {
    SetUserName(std::forward<UserNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the policy document.</p> <p>This parameter allows (through its <a
   * href="http://wikipedia.org/wiki/regex">regex pattern</a>) a string of characters
   * consisting of upper and lowercase alphanumeric characters with no spaces. You
   * can also include any of the following characters: _+=,.@-</p>
   */
  inline const Aws::String& GetPolicyName() const { return m_policyName; }
  inline bool PolicyNameHasBeenSet() const { return m_policyNameHasBeenSet; }
  template <typename PolicyNameT = Aws::String>
  void SetPolicyName(PolicyNameT&& value) {
    m_policyNameHasBeenSet = true;
    m_policyName = std::forward<PolicyNameT>(value);
  }
  template <typename PolicyNameT = Aws::String>
  PutUserPolicyRequest& WithPolicyName(PolicyNameT&& value) {
    SetPolicyName(std::forward<PolicyNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The policy document.</p> <p>You must provide policies in JSON format in IAM.
   * However, for CloudFormation templates formatted in YAML, you can provide the
   * policy in JSON or YAML format. CloudFormation always converts a YAML policy to
   * JSON format before submitting it to IAM.</p> <p>The <a
   * href="http://wikipedia.org/wiki/regex">regex pattern</a> used to validate this
   * parameter is a string of characters consisting of the following:</p> <ul> <li>
   * <p>Any printable ASCII character ranging from the space character
   * (<code>\u0020</code>) through the end of the ASCII character range</p> </li>
   * <li> <p>The printable characters in the Basic Latin and Latin-1 Supplement
   * character set (through <code>\u00FF</code>)</p> </li> <li> <p>The special
   * characters tab (<code>\u0009</code>), line feed (<code>\u000A</code>), and
   * carriage return (<code>\u000D</code>)</p> </li> </ul>
   */
  inline const Aws::String& GetPolicyDocument() const { return m_policyDocument; }
  inline bool PolicyDocumentHasBeenSet() const { return m_policyDocumentHasBeenSet; }
  template <typename PolicyDocumentT = Aws::String>
  void SetPolicyDocument(PolicyDocumentT&& value) {
    m_policyDocumentHasBeenSet = true;
    m_policyDocument = std::forward<PolicyDocumentT>(value);
  }
  template <typename PolicyDocumentT = Aws::String>
  PutUserPolicyRequest& WithPolicyDocument(PolicyDocumentT&& value) {
    SetPolicyDocument(std::forward<PolicyDocumentT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_userName;
  bool m_userNameHasBeenSet = false;

  Aws::String m_policyName;
  bool m_policyNameHasBeenSet = false;

  Aws::String m_policyDocument;
  bool m_policyDocumentHasBeenSet = false;
};

}  // namespace Model
}  // namespace IAM
}  // namespace Aws
